unit RTLabel;

{ Unit: RTLabel 1.11                              (C)1996-1998 by DuNo Electronics I/S }
{ ------------------------------------------------------------------------------------ }
{ Created:  Mar 22 1996                                                                }
{ Modified: May 15 1998                                                                }
{ Author:   Johnny Norre, DuNo Electronics I/S                                         }
{ Status:   Freeware:  The message "Portions copyright DuNo Electronics I/S" must be   }
{                      included somewhere in the program (e.g. the about box or the    }
{                      help file) or in the documentation (e.g. the manual or the      }
{                      "read me" file). The copyright notice in the source code must   }
{                      not be changed.                                                 }
{ ------------------------------------------------------------------------------------ }
{ RTLabel provides a label and a button on which the text can be rotated. Also         }
{ provides a font object, which can be rotated.                                        }
{ ------------------------------------------------------------------------------------ }
{ Wish list: Make a QRRTLabel to use on a QuickReport.                                 }
{ ------------------------------------------------------------------------------------ }
{ Version   Date    Description                                                        }
{   0.0   22.03.96  RTLabel basic implementation.                                      }
{   0.1   24.04.96  RTButton basic implementation.                                     }
{   1.0   29.04.96  Version 0.2 promoted to release version.                           }
{   1.1   09.07.96  DUNOSAFE problem corrected                                         }
{   1.11  15.05.98  Dependency of DUNOSAFE removed.                                    }
{ ------------------------------------------------------------------------------------ }

interface

{$M+}

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  StdCtrls;

const
  PALETTEPAGE = 'Additional';  { Where to install the components }

type
  { Our extended font data information structure }
  TRTFontData = record
    Height : integer;
    Pitch : TFontPitch;
    Style : TFontStyles;
    Name : TFontName;
    Escapement : integer;
  end;
  TRTFont = class(TFont)  { Derive from the TFont object }
  private
    FontData : TRTFontData; { This is a local copy of the TFontData structure maintained }
                            { by the TFont object - Delphi destroys this data when a new }
                            { font handle is assigned.                                   }
    procedure MakeNewFont;  { Creates a new font and assigns it to the property handle }
  protected
    procedure SetEscapement(Value : integer);  
    function  NewGetSize : integer;
    procedure NewSetHeight(Value : integer);
    procedure NewSetName(Value : TFontName);
    procedure NewSetPitch(Value : TFontPitch);
    procedure NewSetSize(Value : integer);
    procedure NewSetStyle(Value : TFontStyles);
  public
    constructor Create;
    procedure Assign(Source : TPersistent); override;
    property Height: Integer read FontData.Height write NewSetHeight;
    property Name: TFontName read FontData.Name write NewSetName;
    property Pitch: TFontPitch read FontData.Pitch write NewSetPitch default fpDefault;
    property Size: Integer read NewGetSize write NewSetSize stored False;
    property Style: TFontStyles read FontData.Style write NewSetStyle;
    property Escapement : integer read FontData.Escapement write SetEscapement default 0;
  end;
  TRTLabel = class(TCustomLabel)
  private
    { Private declarations }
    FEscapement : integer; { to avoid writing a new TRTFont property editor }
    FVerticalCenter : boolean;
    procedure CMFontChanged(var Msg : TMessage); message CM_FONTCHANGED;
    procedure AdjustBounds;
    procedure SetCaption(Value : string);
    function  GetCaption : string;
    procedure SetEscapement(Value : integer);  { This does the dirty work }
    function  GetEscapement : integer;
    procedure SetAutoSize(Value : boolean); override;
    function  GetRTFont : TRTFont;
  protected
    { Protected declarations }
    procedure UpdateRTFont;
    procedure Paint; override;
  public
    { Public declarations }
    FRTFont : TRTFont;
    constructor Create(AOwner : TComponent); override;
    destructor Destroy; override;
    property RTFont : TRTFont read GetRTFont;
  published
    { Published declarations }
    property Align;
    property Alignment;
    property AutoSize write SetAutoSize;
    property Caption : string read GetCaption write SetCaption;
    property Color;
    property DragCursor;
    property DragMode;
    property Enabled;
    property FocusControl;
    property Font;
    property FontEscapement : integer read GetEscapement write SetEscapement;
    property ParentColor;
    property ParentFont;
    property ParentShowHint;
    property PopupMenu;
    property ShowAccelChar;
    property ShowHint;
    property Transparent;
    property Visible;
    property OnClick;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDrag;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
    property OnStartDrag;
  end;
  TRTButton = class(TButton)
  private
    { Private declarations }
    FEscapement : integer; { to avoid writing a new TRTFont property editor }
    FCanvas : TCanvas;
    procedure CMFontChanged(var Msg : TMessage); message CM_FONTCHANGED;
    procedure CNMeasureItem(var Msg : TWMMeasureItem); message CN_MEASUREITEM;
    procedure CNDrawItem(var Msg : TWMDrawItem); message CN_DRAWITEM;
    procedure DrawItem(const DrawItemStruct : TDrawItemStruct);
    procedure SetEscapement(Value : integer);  { This does the dirty work }
    function  GetEscapement : integer;
    function  GetRTFont : TRTFont;
  protected
    { Protected declarations }
    procedure UpdateRTFont;
    procedure CreateParams(var Params : TCreateParams); override;
    procedure SetButtonStyle(ADefault : boolean); override;
  public
    { Public declarations }
    FRTFont : TRTFont;
    constructor Create(AOwner : TComponent); override;
    destructor Destroy; override;
    property RTFont : TRTFont read GetRTFont;
  published
    { Published declarations }
    property FontEscapement : integer read GetEscapement write SetEscapement;
  end;

procedure Register;

implementation

function IsWin95 : boolean;
var
  L : longint;
begin
  L := loword(GetVersion);
  L := (L mod 256) shl 8+L div 256;
  IsWin95 := L >= $35F;
end;

{ RemoveAccelChar removes the accellerator character (&) from a string and }
{ returns the position of the character in "AccelPos"                      }
function RemoveAccelChar(TxtStr : string; var AccelPos : integer) : string;
var
  S : string;
  dN,N : integer;
begin
  S := TxtStr;
  TxtStr := '';
  AccelPos := 0;
  dN := 0;
  N := pos('&',S);
  while N > 0 do begin
    if N = length(S) then begin
      RemoveAccelChar := copy(S,1,N-1);
      exit
    end;
    if S[N+1] = '&' then begin
      TxtStr := TxtStr + copy(S,1,N);
      delete(S,1,N+1);
      dN := N+1
    end else
      break;
    N := pos('&',S)
  end;
  if N > 0 then begin
    TxtStr := TxtStr+copy(S,1,N-1);
    S := copy(S,N+1,length(S));
    while pos('&&',S) > 0 do begin
      TxtStr := TxtStr+copy(S,1,pos('&&',S));
      delete(S,1,pos('&&',S)+1);
    end;
    RemoveAccelChar := TxtStr+S;
    AccelPos := dN+N
  end else
    RemoveAccelChar := TxtStr+S
end;


{ ******** TRTFONT ******** }

constructor TRTFont.Create;
begin
  inherited Create;
  FontData.Name := 'MS Sans Serif'; { Because a standard TFont is created with MS Sans Serif } 
  FontData.Height := -11;
  FontData.Pitch := fpDefault;
  FontData.Style := [];
  FontData.Escapement := 0
end;

{ Assign copies the font data from either a Delphi TFont or another TRTFont object }
procedure TRTFont.Assign(Source : TPersistent);
begin
  inherited Assign(Source);
  if Source is TRTFont then begin
    FontData.Name := TRTFont(Source).FontData.Name;
    FontData.Height := TRTFont(Source).FontData.Height;
    FontData.Pitch := TRTFont(Source).FontData.Pitch;
    FontData.Style := TRTFont(Source).FontData.Style;
    FontData.Escapement := TRTFont(Source).FontData.Escapement
  end else if Source is TFont then begin
    { As the standard TFont resets it's values to defaults when assigning a new handle, }
    { we only update the FontData structure if the Font property is different from the  }
    { defaults.                                                                         }
    if (TFont(Source).Name <> 'MS Sans Serif') or (TFont(Source).Height = 0) or
       (TFont(Source).Pitch <> fpDefault) or (TFont(Source).Style <> []) then begin
      FontData.Name := TFont(Source).Name;
      FontData.Height := TFont(Source).Height;
      FontData.Pitch := TFont(Source).Pitch;
      FontData.Style := TFont(Source).Style
    end
  end
end;

procedure TRTFont.MakeNewFont;
var
  LogFont : TLogFont;
begin
  with LogFont do begin
    lfHeight := FontData.Height;
    lfWidth := 0; { have font mapper choose }
    lfEscapement := FontData.Escapement; { ** set to our escapement ** }
    lfOrientation := 0; { no rotation }
    if fsBold in FontData.Style then
      lfWeight := FW_BOLD
    else
      lfWeight := FW_NORMAL;
    lfItalic := Byte(fsItalic in FontData.Style);
    lfUnderline := Byte(fsUnderline in FontData.Style);
    lfStrikeOut := Byte(fsStrikeOut in FontData.Style);
    lfCharSet := DEFAULT_CHARSET;
    StrPCopy(lfFaceName, FontData.Name);
    lfQuality := DEFAULT_QUALITY;
    { Everything else as default }
    lfOutPrecision := OUT_DEFAULT_PRECIS;
    lfClipPrecision := CLIP_DEFAULT_PRECIS;
    case FontData.Pitch of
      fpVariable: lfPitchAndFamily := VARIABLE_PITCH;
      fpFixed: lfPitchAndFamily := FIXED_PITCH;
    else
      lfPitchAndFamily := DEFAULT_PITCH;
    end
  end;
  Handle := CreateFontIndirect(LogFont)
end;

function TRTFont.NewGetSize: Integer;
begin
  Result := -MulDiv(FontData.Height, 72, PixelsPerInch)
end;

procedure TRTFont.NewSetHeight(Value : integer);
begin
  if Value <> FontData.Height then begin
    FontData.Height:= Value;
    MakeNewFont
  end
end;

procedure TRTFont.NewSetName(Value : TFontName);
begin
  if Value <> FontData.Name then begin
    FontData.Name := Value;
    MakeNewFont
  end
end;

procedure TRTFont.NewSetPitch(Value : TFontPitch);
begin
  if Value <> FontData.Pitch then begin
    FontData.Pitch := Value;
    MakeNewFont
  end
end;

procedure TRTFont.NewSetSize(Value: Integer);
begin
  FontData.Height := -MulDiv(Value, PixelsPerInch, 72);
end;

procedure TRTFont.NewSetStyle(Value : TFontStyles);
begin
  if Value <> FontData.Style then begin
    FontData.Style := Value;
    MakeNewFont
  end
end;

procedure TRTFont.SetEscapement;
begin
  if Value <> FontData.Escapement then begin
    FontData.Escapement := Value;
    MakeNewFont
  end
end;


{ ******** TRTLABEL ******** }

constructor TRTLabel.Create(AOwner : TComponent);
begin
  inherited Create(AOwner);
  FEscapement := 0
end;

destructor TRTLabel.Destroy;
begin
  RTFont.Free;
  inherited Destroy
end;

function TRTLabel.GetRTFont;
begin
  if not assigned(FRTFont) then begin
    FRTFont := TRTFont.Create;
    FRTFont.Assign(Font)
  end;
  Result := FRTFont
end;

procedure TRTLabel.Paint;
var
  Rect: TRect;
  Text : string;
  FontSize,
  FontSize2 : TSize;
  X,Y,
  cX,cY : integer;
  AccelPos : integer;
begin
  with Canvas do begin
    if not Transparent then begin
      Brush.Color := Self.Color;
      Brush.Style := bsSolid;
      FillRect(ClientRect)
    end;
    Brush.Style := bsClear;
    Rect := ClientRect;
    Canvas.Font := RTFont;
    if not Enabled then
      Canvas.Font.Color := clGrayText;
    if ShowAccelChar then
      Text := RemoveAccelChar(Caption,AccelPos)
    else
      Text := Caption;
    GetTextExtentPoint(Canvas.Handle,@Text[1],length(Text),FontSize);
    cX := trunc(FontSize.cX*cos(FEscapement/3600*2*pi)+FontSize.cY*sin(FEscapement/3600*2*pi)) div 2;
    cy := trunc(FontSize.cY*sin((FEscapement+900)/3600*2*pi)+FontSize.cX*cos((FEscapement+900)/3600*2*pi)) div 2;
    case Alignment of
      taCenter : { Calculate text placement to match center of the text with the center of the client rect }
        X := (Rect.Right div 2) - cX;
      taLeftJustify :
        X := (FontSize.cX div 2) - cx;
      taRightJustify :
        X := (Rect.Right - (FontSize.cX div 2)) - cx;
    end;
    Y := (Rect.Bottom div 2) - cY;
    TextOut(X,Y,Text);
    if ShowAccelChar and (AccelPos > 0) then begin
      if AccelPos = 1 then
        FontSize.cX := 0
      else
        GetTextExtentPoint(Canvas.Handle,@Text[1],pred(AccelPos),FontSize);
      GetTextExtentPoint(Canvas.Handle,@Text[AccelPos],1,FontSize2);
      MoveTo(X+trunc(sin(FEscapement/3600*2*pi)*FontSize.cY)+trunc(sin((FEscapement+900)/3600*2*pi)*FontSize.cX),
             Y+trunc(cos(FEscapement/3600*2*pi)*FontSize.cY)+trunc(cos((FEscapement+900)/3600*2*pi)*FontSize.cX));
      LineTo(X+trunc(sin(FEscapement/3600*2*pi)*FontSize.cY)+trunc(sin((FEscapement+900)/3600*2*pi)*
             (FontSize.cX+FontSize2.cX)),
             Y+trunc(cos(FEscapement/3600*2*pi)*FontSize.cY)+trunc(cos((FEscapement+900)/3600*2*pi)*
             (FontSize.cX+FontSize2.cX)));
    end
  end
end;

procedure TRTLabel.UpdateRTFont;
begin
  if RTFont.Escapement <> FEscapement then
    RTFont.Escapement := FEscapement;
  if RTFont.Name <> Font.Name then
    RTFont.Name := Font.Name;
  if RTFont.Height <> Font.Height  then
    RTFont.Height := Font.Height;
  if RTFont.Pitch <> Font.Pitch then
    RTFont.Pitch := Font.Pitch;
  if RTFont.Style <> Font.Style then
    RTFont.Style := Font.Style;
  if RTFont.Color <> Font.Color then
    RTFont.Color := Font.Color;
  RTFont.Changed;
end;

procedure TRTLabel.CMFontChanged;
begin
  UpdateRTFont;
  inherited;
  if AutoSize then
    AdjustBounds
end;

procedure TRTLabel.SetCaption;
begin
  inherited Caption := Value;
  if AutoSize then
    AdjustBounds
end;

function  TRTLabel.GetCaption;
begin
  Result := inherited Caption
end;

procedure TRTLabel.SetEscapement;
begin
  if Value <> FEscapement then begin
    FEscapement := Value;
    UpdateRTFont;
    Invalidate
  end
end;

procedure TRTLabel.AdjustBounds;
var
  FontSize : TSize;
  Text : string;
begin
  Canvas.Font := RTFont;
  Text := Caption;
  GetTextExtentPoint(Canvas.Handle,@Text[1],length(Text),FontSize);
  if FontSize.cX > FontSize.cY then
    SetBounds(Left,Top,FontSize.cX,FontSize.cX)
  else
    SetBounds(Left,Top,FontSize.cY,FontSize.cY)
end;

procedure TRTLabel.SetAutoSize;
begin
  inherited SetAutoSize(value);
  if Value then
    AdjustBounds
end;

function TRTLabel.GetEscapement;
begin
  GetEscapement := FEscapement
end;


{ ******** TRTBUTTON ******** }
constructor TRTButton.Create(AOwner : TComponent);
begin
  inherited Create(AOwner);
  FCanvas := TCanvas.Create;
  FEscapement := 0
end;

destructor TRTButton.Destroy;
begin
  RTFont.Free;
  FCanvas.Free;
  inherited Destroy
end;

procedure TRTButton.CreateParams;
begin
  inherited CreateParams(Params);
  with Params do
    Style := Style or BS_OWNERDRAW;
end;

procedure TRTButton.SetButtonStyle(ADefault : boolean);
var
  Style : word;
begin
  if HandleAllocated then begin
    if ADefault then
      Style := BS_DEFPUSHBUTTON or BS_OWNERDRAW
    else
      Style := BS_PUSHBUTTON or BS_OWNERDRAW;
    if GetWindowLong(Handle,GWL_STYLE) and $000F <> Style then
      SendMessage(Handle,BM_SETSTYLE,Style,1);
  end
end;

function TRTButton.GetRTFont;
begin
  if not assigned(FRTFont) then begin
    FRTFont := TRTFont.Create;
    FRTFont.Assign(Font)
  end;
  Result := FRTFont
end;

procedure TRTButton.CNMeasureItem;
begin
  with Msg.MeasureItemStruct^ do begin
    itemWidth := Width;
    itemHeight := Height
  end
end;

procedure TRTButton.CNDrawItem;
begin
  DrawItem(Msg.DrawItemStruct^)
end;

procedure TRTButton.DrawItem;
var
  Rect: TRect;
  Text : string;
  FontSize,
  FontSize2 : TSize;
  X,Y,
  cX,cY : integer;
  IsDown,
  IsFocused : boolean;
  AccelPos : integer;
begin
  FCanvas.Handle := DrawItemStruct.hDC;
  IsDown := DrawItemStruct.itemState and ODS_SELECTED <> 0;
  IsFocused := DrawItemStruct.itemState and ODS_FOCUS <> 0;
  with FCanvas do begin
    if IsWin95 then begin
      Brush.Color := Self.Color;
      FillRect(ClientRect);
    end;
    { Draw frame }
    Rect := ClientRect;
    if Default then begin
      Brush.Color := clWindowFrame;
      Pen.Color := clWindowFrame;
      if IsWin95 then
        FrameRect(Rect)
      else begin
        Brush.Color := clBtnFace;
        RoundRect(Rect.Left,Rect.Top,Rect.Right,Rect.Bottom,2,2)
      end;
      InflateRect(Rect,-1,-1)
    end;
    Brush.Color := clBtnShadow;
    Pen.Color := clBtnShadow;
    if IsFocused then begin
      Brush.Color := clWindowFrame;
      Pen.Color := clWindowFrame;
    end;
    Brush.Style := bsSolid;
    if IsWin95 then
      FrameRect(Rect)
    else begin
      Brush.Color := clBtnFace;
      RoundRect(Rect.Left,Rect.Top,Rect.Right,Rect.Bottom,2,2)
    end;
    InflateRect(Rect,-1,-1);
    if IsDown then
      Pen.Color := clBtnShadow
    else
      Pen.Color := clWhite;
    with Rect do begin
      MoveTo(Left,pred(Bottom));
      LineTo(Left,Top);
      LineTo(pred(Right),Top);
      MoveTo(Left+1,pred(Bottom));
      LineTo(Left+1,Top+1);
      LineTo(pred(Right),Top+1);
    end;
    if not IsDown then begin
      Pen.Color := clBtnShadow;
      with Rect do begin
        MoveTo(pred(Right),top);
        LineTo(pred(Right),pred(Bottom));
        LineTo(left,pred(Bottom));
        MoveTo(pred(Right)-1,top+1);
        LineTo(pred(Right)-1,pred(Bottom)-1);
        LineTo(left+1,pred(Bottom)-1);
      end;
    end;
    Rect := ClientRect;
    Brush.Style := bsClear;
    FCanvas.Font := RTFont;
    if not Enabled then
      FCanvas.Font.Color := clGrayText;
    Text := RemoveAccelChar(Caption,AccelPos);
    GetTextExtentPoint(FCanvas.Handle,@Text[1],length(Text),FontSize);
    cX := trunc(FontSize.cX*cos(FEscapement/3600*2*pi)+FontSize.cY*sin(FEscapement/3600*2*pi)) div 2;
    cy := trunc(FontSize.cY*sin((FEscapement+900)/3600*2*pi)+FontSize.cX*cos((FEscapement+900)/3600*2*pi)) div 2;
    X := (Rect.Right div 2) - cX;
    Y := (Rect.Bottom div 2) - cY;
    if IsDown then begin
      X := X+2;
      Y := Y+2
    end;
    Pen.Color := Font.Color;
    TextOut(X,Y,Text);
    Brush.Style := bsSolid;
    if IsFocused then begin
      InflateRect(Rect,-5,-5);
      DrawFocusRect(Rect);
    end;
    if AccelPos > 0 then begin
      if AccelPos = 1 then
        FontSize.cX := 0
      else
        GetTextExtentPoint(FCanvas.Handle,@Text[1],pred(AccelPos),FontSize);
      GetTextExtentPoint(FCanvas.Handle,@Text[AccelPos],1,FontSize2);
      MoveTo(X+trunc(sin(FEscapement/3600*2*pi)*FontSize.cY)+trunc(sin((FEscapement+900)/3600*2*pi)*FontSize.cX),
             Y+trunc(cos(FEscapement/3600*2*pi)*FontSize.cY)+trunc(cos((FEscapement+900)/3600*2*pi)*FontSize.cX));
      LineTo(X+trunc(sin(FEscapement/3600*2*pi)*FontSize.cY)+trunc(sin((FEscapement+900)/3600*2*pi)*
             (FontSize.cX+FontSize2.cX)),
             Y+trunc(cos(FEscapement/3600*2*pi)*FontSize.cY)+trunc(cos((FEscapement+900)/3600*2*pi)*
             (FontSize.cX+FontSize2.cX)));
    end
  end;
  FCanvas.Handle := 0
end;

procedure TRTButton.UpdateRTFont;
begin
  if RTFont.Escapement <> FEscapement then
    RTFont.Escapement := FEscapement;
  if RTFont.Name <> Font.Name {'MS Sans Serif')} then
    RTFont.Name := Font.Name;
  if RTFont.Height <> Font.Height  then
    RTFont.Height := Font.Height;
  if RTFont.Pitch <> Font.Pitch then
    RTFont.Pitch := Font.Pitch;
  if RTFont.Style <> Font.Style then
    RTFont.Style := Font.Style;
  if RTFont.Color <> Font.Color then
    RTFont.Color := Font.Color;
  RTFont.Changed;
end;

procedure TRTButton.CMFontChanged;
begin
  UpdateRTFont;
  inherited;
end;

procedure TRTButton.SetEscapement;
begin
  if Value <> FEscapement then begin
    FEscapement := Value;
    UpdateRTFont;
    Invalidate
  end
end;

function TRTButton.GetEscapement;
begin
  GetEscapement := FEscapement
end;

procedure Register;
begin
  RegisterComponents(PALETTEPAGE, [TRTLabel,TRTButton]);
end;

end.
